#include "burnint.h"
#include "burn_sound.h"
#include "burn_ym3812b.h"
#include "zet.h"
#include "fmopl.h"
#include "zet2.h"


#define MAX_YM3812b	2

// Timer Related

#define MAX_TIMER_VALUE ((1 << 30) - 65536)

static double dTimeYM3812b;									// Time elapsed since the emulated machine was started

static INT32 nTimerCount[2], nTimerStart[2];

// Callbacks
static INT32 (*pTimerOverCallback)(INT32, INT32);
static double (*pTimerTimeCallback)();

static INT32 nCPUClockspeed = 0;
static INT32 (*pCPUTotalCycles)() = NULL;
static INT32 (*pCPURun)(INT32) = NULL;
static void (*pCPURunEnd)() = NULL;

// ---------------------------------------------------------------------------
// Running time

static double BurnTimerTimeCallbackDummy()
{
	return 0.0;
}

extern "C" double BurnTimerGetTimeYM3812b()
{
	return dTimeYM3812b + pTimerTimeCallback();
}

// ---------------------------------------------------------------------------
// Update timers

static INT32 nTicksTotal, nTicksDone, nTicksExtra;

INT32 BurnTimerUpdateYM3812b(INT32 nCycles)
{
	INT32 nIRQStatus = 0;

	nTicksTotal = MAKE_TIMER_TICKS(nCycles, nCPUClockspeed);

	while (nTicksDone < nTicksTotal) {
		INT32 nTimer, nCyclesSegment, nTicksSegment;

		// Determine which timer fires first
		if (nTimerCount[0] <= nTimerCount[1]) {
			nTicksSegment = nTimerCount[0];
		} else {
			nTicksSegment = nTimerCount[1];
		}
		if (nTicksSegment > nTicksTotal) {
			nTicksSegment = nTicksTotal;
		}

		nCyclesSegment = MAKE_CPU_CYLES(nTicksSegment + nTicksExtra, nCPUClockspeed);

		pCPURun(nCyclesSegment - pCPUTotalCycles());

		nTicksDone = MAKE_TIMER_TICKS(pCPUTotalCycles() + 1, nCPUClockspeed) - 1;

		nTimer = 0;
		if (nTicksDone >= nTimerCount[0]) {
			if (nTimerStart[0] == MAX_TIMER_VALUE) {
				nTimerCount[0] = MAX_TIMER_VALUE;
			} else {
				nTimerCount[0] += nTimerStart[0];
			}
			nTimer |= 1;
		}
		if (nTicksDone >= nTimerCount[1]) {
			if (nTimerStart[1] == MAX_TIMER_VALUE) {
				nTimerCount[1] = MAX_TIMER_VALUE;
			} else {
				nTimerCount[1] += nTimerStart[1];
			}
			nTimer |= 2;
		}
		if (nTimer & 1) {
			nIRQStatus |= pTimerOverCallback(0, 0);
		}
		if (nTimer & 2) {
			nIRQStatus |= pTimerOverCallback(0, 1);
		}
	}

	return nIRQStatus;
}

void BurnTimerEndFrameYM3812b(INT32 nCycles)
{
	INT32 nTicks = MAKE_TIMER_TICKS(nCycles, nCPUClockspeed);

	BurnTimerUpdateYM3812b(nCycles);

	if (nTimerCount[0] < MAX_TIMER_VALUE) {
		nTimerCount[0] -= nTicks;
	}
	if (nTimerCount[1] < MAX_TIMER_VALUE) {
		nTimerCount[1] -= nTicks;
	}

	nTicksDone -= nTicks;
	if (nTicksDone < 0) {
		nTicksDone = 0;
	}
}

void BurnTimerUpdateEndYM3812b()
{
	pCPURunEnd();

	nTicksTotal = 0;
}

void BurnOPLTimerCallbackYM3812b(INT32 c, double period)
{
	pCPURunEnd();

	if (period == 0.0) {
		nTimerCount[c] = MAX_TIMER_VALUE;
		return;
	}

	nTimerCount[c]  = (INT32)(period * (double)TIMER_TICKS_PER_SECOND);
	nTimerCount[c] += MAKE_TIMER_TICKS(pCPUTotalCycles(), nCPUClockspeed);
}

void BurnTimerScanYM3812b(INT32 nAction, INT32* pnMin)
{
	if (pnMin && *pnMin < 0x029521) {
		*pnMin = 0x029521;
	}

	if (nAction & ACB_DRIVER_DATA) {
		SCAN_VAR(nTimerCount);
		SCAN_VAR(nTimerStart);
		SCAN_VAR(dTimeYM3812b);

		SCAN_VAR(nTicksDone);
	}
}

void BurnTimerExitYM3812b()
{
	nCPUClockspeed = 0;
	pCPUTotalCycles = NULL;
	pCPURun = NULL;
	pCPURunEnd = NULL;

	return;
}

void BurnTimerResetYM3812b()
{
	nTimerCount[0] = nTimerCount[1] = MAX_TIMER_VALUE;
	nTimerStart[0] = nTimerStart[1] = MAX_TIMER_VALUE;

	dTimeYM3812b = 0.0;

	nTicksDone = 0;
}

INT32 BurnTimerInitYM3812b(INT32 (*pOverCallback)(INT32, INT32), double (*pTimeCallback)())
{
	BurnTimerExitYM3812b();

	pTimerOverCallback = pOverCallback;
	pTimerTimeCallback = pTimeCallback ? pTimeCallback : BurnTimerTimeCallbackDummy;

	BurnTimerResetYM3812b();

	return 0;
}


INT32 BurnTimerAttachZetYM3812b(INT32 nClockspeed)
{
	nCPUClockspeed = nClockspeed;
	pCPUTotalCycles = ZetTotalCycles;
	pCPURun = ZetRun;
	pCPURunEnd = ZetRunEnd;

	nTicksExtra = MAKE_TIMER_TICKS(1, nCPUClockspeed) - 1;

	return 0;
}

INT32 BurnTimerAttachZet2YM3812b(INT32 nClockspeed)
{
	nCPUClockspeed = nClockspeed;
	pCPUTotalCycles = Zet2TotalCycles;
	pCPURun = Zet2Run;
	pCPURunEnd = Zet2RunEnd;

	nTicksExtra = MAKE_TIMER_TICKS(1, nCPUClockspeed) - 1;

	return 0;
}

// Sound Related

void (*BurnYM3812bUpdate)(INT16* pSoundBuf, INT32 nSegmentEnd);

static INT32 (*BurnYM3812bStreamCallback)(INT32 nSoundRate);

static INT32 nBurnYM3812bSoundRate;

static INT16* pBuffer;
static INT16* pYM3812bBuffer[2 * MAX_YM3812b];

static INT32 nYM3812bPosition;

static UINT32 nSampleSize;
static INT32 nFractionalPosition;

static INT32 nNumChips = 0;
static INT32 bYM3812bAddSignal;

static double YM3812bVolumes[1 * MAX_YM3812b];
static INT32 YM3812bRouteDirs[1 * MAX_YM3812b];

// ----------------------------------------------------------------------------
// Dummy functions

static void YM3812bUpdateDummy(INT16* , INT32)
{
	return;
}

static INT32 YM3812bStreamCallbackDummy(INT32)
{
	return 0;
}

// ----------------------------------------------------------------------------
// Execute YM3812 for part of a frame

static void YM3812bRender(INT32 nSegmentLength)
{
	if (nYM3812bPosition >= nSegmentLength) {
		return;
	}

	nSegmentLength -= nYM3812bPosition;

	YM3812bUpdateOne(0, pBuffer + 0 * 4096 + 4 + nYM3812bPosition, nSegmentLength);
	
	if (nNumChips > 1) {
		YM3812bUpdateOne(1, pBuffer + 1 * 4096 + 4 + nYM3812bPosition, nSegmentLength);
	}

	nYM3812bPosition += nSegmentLength;
}

// ----------------------------------------------------------------------------
// Update the sound buffer

static void YM3812bUpdateResample(INT16* pSoundBuf, INT32 nSegmentEnd)
{

	INT32 nSegmentLength = nSegmentEnd;
	INT32 nSamplesNeeded = nSegmentEnd * nBurnYM3812bSoundRate / nBurnSoundRate + 1;

	if (nSamplesNeeded < nYM3812bPosition) {
		nSamplesNeeded = nYM3812bPosition;
	}

	if (nSegmentLength > nBurnSoundLen) {
		nSegmentLength = nBurnSoundLen;
	}
	nSegmentLength <<= 1;

	YM3812bRender(nSamplesNeeded);

	pYM3812bBuffer[0] = pBuffer + 0 * 4096 + 4;
	if (nNumChips > 1) {
		pYM3812bBuffer[1] = pBuffer + 1 * 4096 + 4;
	}

	for (INT32 i = (nFractionalPosition & 0xFFFF0000) >> 15; i < nSegmentLength; i += 2, nFractionalPosition += nSampleSize) {
		INT32 nLeftSample[4] = {0, 0, 0, 0};
		INT32 nRightSample[4] = {0, 0, 0, 0};
		INT32 nTotalLeftSample, nTotalRightSample;
		
		if ((YM3812bRouteDirs[0 + BURN_SND_YM3812b_ROUTE] & BURN_SND_ROUTE_LEFT) == BURN_SND_ROUTE_LEFT) {
			nLeftSample[0] += (INT32)(pYM3812bBuffer[0][(nFractionalPosition >> 16) - 3] * YM3812bVolumes[0 + BURN_SND_YM3812b_ROUTE]);
			nLeftSample[1] += (INT32)(pYM3812bBuffer[0][(nFractionalPosition >> 16) - 2] * YM3812bVolumes[0 + BURN_SND_YM3812b_ROUTE]);
			nLeftSample[2] += (INT32)(pYM3812bBuffer[0][(nFractionalPosition >> 16) - 1] * YM3812bVolumes[0 + BURN_SND_YM3812b_ROUTE]);
			nLeftSample[3] += (INT32)(pYM3812bBuffer[0][(nFractionalPosition >> 16) - 0] * YM3812bVolumes[0 + BURN_SND_YM3812b_ROUTE]);
		}
		if ((YM3812bRouteDirs[0 + BURN_SND_YM3812b_ROUTE] & BURN_SND_ROUTE_RIGHT) == BURN_SND_ROUTE_RIGHT) {
			nRightSample[0] += (INT32)(pYM3812bBuffer[0][(nFractionalPosition >> 16) - 3] * YM3812bVolumes[0 + BURN_SND_YM3812b_ROUTE]);
			nRightSample[1] += (INT32)(pYM3812bBuffer[0][(nFractionalPosition >> 16) - 2] * YM3812bVolumes[0 + BURN_SND_YM3812b_ROUTE]);
			nRightSample[2] += (INT32)(pYM3812bBuffer[0][(nFractionalPosition >> 16) - 1] * YM3812bVolumes[0 + BURN_SND_YM3812b_ROUTE]);
			nRightSample[3] += (INT32)(pYM3812bBuffer[0][(nFractionalPosition >> 16) - 0] * YM3812bVolumes[0 + BURN_SND_YM3812b_ROUTE]);
		}
		
		if (nNumChips > 1) {
			if ((YM3812bRouteDirs[1 + BURN_SND_YM3812b_ROUTE] & BURN_SND_ROUTE_LEFT) == BURN_SND_ROUTE_LEFT) {
				nLeftSample[0] += (INT32)(pYM3812bBuffer[1][(nFractionalPosition >> 16) - 3] * YM3812bVolumes[1 + BURN_SND_YM3812b_ROUTE]);
				nLeftSample[1] += (INT32)(pYM3812bBuffer[1][(nFractionalPosition >> 16) - 2] * YM3812bVolumes[1 + BURN_SND_YM3812b_ROUTE]);
				nLeftSample[2] += (INT32)(pYM3812bBuffer[1][(nFractionalPosition >> 16) - 1] * YM3812bVolumes[1 + BURN_SND_YM3812b_ROUTE]);
				nLeftSample[3] += (INT32)(pYM3812bBuffer[1][(nFractionalPosition >> 16) - 0] * YM3812bVolumes[1 + BURN_SND_YM3812b_ROUTE]);
			}
			if ((YM3812bRouteDirs[1 + BURN_SND_YM3812b_ROUTE] & BURN_SND_ROUTE_RIGHT) == BURN_SND_ROUTE_RIGHT) {
				nRightSample[0] += (INT32)(pYM3812bBuffer[1][(nFractionalPosition >> 16) - 3] * YM3812bVolumes[1 + BURN_SND_YM3812b_ROUTE]);
				nRightSample[1] += (INT32)(pYM3812bBuffer[1][(nFractionalPosition >> 16) - 2] * YM3812bVolumes[1 + BURN_SND_YM3812b_ROUTE]);
				nRightSample[2] += (INT32)(pYM3812bBuffer[1][(nFractionalPosition >> 16) - 1] * YM3812bVolumes[1 + BURN_SND_YM3812b_ROUTE]);
				nRightSample[3] += (INT32)(pYM3812bBuffer[1][(nFractionalPosition >> 16) - 0] * YM3812bVolumes[1 + BURN_SND_YM3812b_ROUTE]);
			}
		}
		
		nTotalLeftSample = INTERPOLATE4PS_16BIT((nFractionalPosition >> 4) & 0x0fff, nLeftSample[0], nLeftSample[1], nLeftSample[2], nLeftSample[3]);
		nTotalRightSample = INTERPOLATE4PS_16BIT((nFractionalPosition >> 4) & 0x0fff, nRightSample[0], nRightSample[1], nRightSample[2], nRightSample[3]);
		
		nTotalLeftSample = BURN_SND_CLIP(nTotalLeftSample);
		nTotalRightSample = BURN_SND_CLIP(nTotalRightSample);
			
		if (bYM3812bAddSignal) {
			pSoundBuf[i + 0] = BURN_SND_CLIP(pSoundBuf[i + 0] + nTotalLeftSample);
			pSoundBuf[i + 1] = BURN_SND_CLIP(pSoundBuf[i + 1] + nTotalRightSample);
		} else {
			pSoundBuf[i + 0] = nTotalLeftSample;
			pSoundBuf[i + 1] = nTotalRightSample;
		}
	}

	if (nSegmentEnd >= nBurnSoundLen) {
		INT32 nExtraSamples = nSamplesNeeded - (nFractionalPosition >> 16);

		for (INT32 i = -4; i < nExtraSamples; i++) {
			pYM3812bBuffer[0][i] = pYM3812bBuffer[0][(nFractionalPosition >> 16) + i];
			if (nNumChips > 1) {
				pYM3812bBuffer[1][i] = pYM3812bBuffer[1][(nFractionalPosition >> 16) + i];
			}
		}

		nFractionalPosition &= 0xFFFF;

		nYM3812bPosition = nExtraSamples;
	}
}

static void YM3812bUpdateNormal(INT16* pSoundBuf, INT32 nSegmentEnd)
{

	INT32 nSegmentLength = nSegmentEnd;

	if (nSegmentEnd < nYM3812bPosition) {
		nSegmentEnd = nYM3812bPosition;
	}

	if (nSegmentLength > nBurnSoundLen) {
		nSegmentLength = nBurnSoundLen;
	}

	YM3812bRender(nSegmentEnd);

	pYM3812bBuffer[0] = pBuffer + 4 + 0 * 4096;
	pYM3812bBuffer[1] = pBuffer + 4 + 1 * 4096;

	for (INT32 n = nFractionalPosition; n < nSegmentLength; n++) {
		INT32 nLeftSample = 0, nRightSample = 0;
		
		if ((YM3812bRouteDirs[0 + BURN_SND_YM3812b_ROUTE] & BURN_SND_ROUTE_LEFT) == BURN_SND_ROUTE_LEFT) {
			nLeftSample += (INT32)(pYM3812bBuffer[0][n] * YM3812bVolumes[0 + BURN_SND_YM3812b_ROUTE]);
		}
		if ((YM3812bRouteDirs[0 + BURN_SND_YM3812b_ROUTE] & BURN_SND_ROUTE_RIGHT) == BURN_SND_ROUTE_RIGHT) {
			nRightSample += (INT32)(pYM3812bBuffer[0][n] * YM3812bVolumes[0 + BURN_SND_YM3812b_ROUTE]);
		}
		
		if (nNumChips > 1) {
			if ((YM3812bRouteDirs[1 + BURN_SND_YM3812b_ROUTE] & BURN_SND_ROUTE_LEFT) == BURN_SND_ROUTE_LEFT) {
				nLeftSample += (INT32)(pYM3812bBuffer[1][n] * YM3812bVolumes[1 + BURN_SND_YM3812b_ROUTE]);
			}
			if ((YM3812bRouteDirs[1 + BURN_SND_YM3812b_ROUTE] & BURN_SND_ROUTE_RIGHT) == BURN_SND_ROUTE_RIGHT) {
				nRightSample += (INT32)(pYM3812bBuffer[1][n] * YM3812bVolumes[1 + BURN_SND_YM3812b_ROUTE]);
			}
		}
		
		nLeftSample = BURN_SND_CLIP(nLeftSample);
		nRightSample = BURN_SND_CLIP(nRightSample);
			
		if (bYM3812bAddSignal) {
			pSoundBuf[(n << 1) + 0] = BURN_SND_CLIP(pSoundBuf[(n << 1) + 0] + nLeftSample);
			pSoundBuf[(n << 1) + 1] = BURN_SND_CLIP(pSoundBuf[(n << 1) + 1] + nRightSample);
		} else {
			pSoundBuf[(n << 1) + 0] = nLeftSample;
			pSoundBuf[(n << 1) + 1] = nRightSample;
		}
	}

	nFractionalPosition = nSegmentLength;

	if (nSegmentEnd >= nBurnSoundLen) {
		INT32 nExtraSamples = nSegmentEnd - nBurnSoundLen;

		for (INT32 i = 0; i < nExtraSamples; i++) {
			pYM3812bBuffer[0][i] = pYM3812bBuffer[0][nBurnSoundLen + i];
			if (nNumChips > 1) {
				pYM3812bBuffer[1][i] = pYM3812bBuffer[1][nBurnSoundLen + i];
			}
		}

		nFractionalPosition = 0;

		nYM3812bPosition = nExtraSamples;

	}
}

// ----------------------------------------------------------------------------
// Callbacks for YM3812 core

void BurnYM3812bUpdateRequest(INT32, INT32)
{

	YM3812bRender(BurnYM3812bStreamCallback(nBurnYM3812bSoundRate));
}

// ----------------------------------------------------------------------------
// Initialisation, etc.

void BurnYM3812bReset()
{
	BurnTimerResetYM3812b();

	for (INT32 i = 0; i < nNumChips; i++) {
		YM3812bResetChip(i);
	}
}

void BurnYM3812bExit()
{

	YM3812bShutdown();

	BurnTimerExitYM3812b();

	if (pBuffer) {
		free(pBuffer);
		pBuffer = NULL;
	}
	
	nNumChips = 0;
	bYM3812bAddSignal = 0;
	
}

INT32 BurnYM3812bInit(INT32 num, INT32 nClockFrequency, OPL_IRQHANDLER IRQCallback, INT32 (*StreamCallback)(INT32), INT32 bAddSignal)
{
	
	if (num > MAX_YM3812b) num = MAX_YM3812b;
	
	BurnTimerInitYM3812b(&YM3812bTimerOver, NULL);

	if (nBurnSoundRate <= 0) {
		BurnYM3812bStreamCallback = YM3812bStreamCallbackDummy;

		BurnYM3812bUpdate = YM3812bUpdateDummy;

		YM3812bInit(num, nClockFrequency, 11025);
		return 0;
	}

	BurnYM3812bStreamCallback = StreamCallback;

	if (nFMInterpolation == 3) {
		// Set YM3812 core samplerate to match the hardware
		nBurnYM3812bSoundRate = nClockFrequency / 72;
		// Bring YM3812 core samplerate within usable range
		while (nBurnYM3812bSoundRate > nBurnSoundRate * 3) {
			nBurnYM3812bSoundRate >>= 1;
		}

		BurnYM3812bUpdate = YM3812bUpdateResample;

		nSampleSize = (UINT32)nBurnYM3812bSoundRate * (1 << 16) / nBurnSoundRate;
		nFractionalPosition = 0;
	} else {
		nBurnYM3812bSoundRate = nBurnSoundRate;

		BurnYM3812bUpdate = YM3812bUpdateNormal;
	}

	YM3812bInit(num, nClockFrequency, nBurnYM3812bSoundRate);
	YM3812bSetIRQHandler(0, IRQCallback, 0);
	YM3812bSetTimerHandler(0, &BurnOPLTimerCallbackYM3812b, 0);
	YM3812bSetUpdateHandler(0, &BurnYM3812bUpdateRequest, 0);

	pBuffer = (INT16*)malloc(4096 * num * sizeof(INT16));
	memset(pBuffer, 0, 4096 * num * sizeof(INT16));

	nYM3812bPosition = 0;

	nFractionalPosition = 0;
	
	nNumChips = num;
	bYM3812bAddSignal = bAddSignal;
	
	// default routes
	YM3812bVolumes[BURN_SND_YM3812b_ROUTE] = 1.00;
	YM3812bRouteDirs[BURN_SND_YM3812b_ROUTE] = BURN_SND_ROUTE_BOTH;
	
	if (nNumChips > 0) {
		YM3812bVolumes[1 + BURN_SND_YM3812b_ROUTE] = 1.00;
		YM3812bRouteDirs[1 + BURN_SND_YM3812b_ROUTE] = BURN_SND_ROUTE_BOTH;
	}

	return 0;
}

void BurnYM3812bSetRoute(INT32 nChip, INT32 nIndex, double nVolume, INT32 nRouteDir)
{
	if (nChip == 0) {
		YM3812bVolumes[nIndex] = nVolume;
		YM3812bRouteDirs[nIndex] = nRouteDir;
	}
	
	if (nChip == 1) {
		YM3812bVolumes[1 + nIndex] = nVolume;
		YM3812bRouteDirs[1 + nIndex] = nRouteDir;
	}
}

void BurnYM3812bScan(INT32 nAction, INT32* pnMin)
{

	BurnTimerScanYM3812b(nAction, pnMin);
//	FMOPLScan(FM_OPL_SAVESTATE_YM3812b, 0, nAction, pnMin);
	
	if (nAction & ACB_DRIVER_DATA) {
		SCAN_VAR(nYM3812bPosition);
	}
}
